<?php
ob_start(); // Iniciar el búfer de salida al inicio del script
session_start();

// ***** IMPORTANTE: ESTAS LÍNEAS ESTÁN AQUÍ PARA DEPURAR. *****
// UNA VEZ QUE FUNCIONE, CÁMBIALAS A: error_reporting(0); ini_set('display_errors', 0);
error_reporting(E_ALL); // Reporta todos los errores
ini_set('display_errors', 1); // Muestra los errores en la pantalla
// *************************************************************

// Incluir el archivo de conexión a la base de datos y GuardianKey
include_once 'db/db_connection.php';
include_once 'files/GuardianKey.php';

// --- Lógica PHP para obtener datos de cursos desde la base de datos ---
$courses_data_from_db = [];
$new_courses_this_month = 0;

try {
    // CAMBIO: Seleccionar también el campo 'poster_url' de la base de datos
    $stmt = $pdo->prepare("SELECT id, title, description, url, poster_url, created_at FROM academy ORDER BY created_at DESC");
    $stmt->execute();
    $courses_db_raw = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Formatear los cursos para el frontend JavaScript
    foreach ($courses_db_raw as $course) {
        $indexed_id = 'course-' . $course['id']; // ID único para el frontend JS
        $courses_data_from_db[$indexed_id] = [
            'id' => $indexed_id,
            'db_id' => $course['id'],
            'titulo' => $course['title'],
            'descripcion' => $course['description'],
            'url' => $course['url'],
            'poster_url' => $course['poster_url'], // CAMBIO: Añadir poster_url
            'created_at' => $course['created_at']
        ];

        // Contar cursos nuevos este mes
        $current_month_year = date('Y-m');
        if (isset($course['created_at'])) {
            $created_month_year = (new DateTime($course['created_at']))->format('Y-m');
            if ($created_month_year === $current_month_year) {
                $new_courses_this_month++;
            }
        }
    }

} catch (PDOException $e) {
    error_log("Error PDO al cargar cursos desde academy.php: " . $e->getMessage());
    // Puedes manejar un mensaje de error para el usuario si es crítico
} catch (Exception $e) {
    error_log("Error general al cargar cursos desde academy.php: " . $e->getMessage());
}

// Si el array de cursos está vacío, convertirlo a un objeto vacío para JSON
if (empty($courses_data_from_db)) {
    $courses_data_from_db = new stdClass();
}

// Codificar los datos a JSON para pasarlos a JavaScript
$courses_json = json_encode($courses_data_from_db);
$new_courses_this_month_json = json_encode($new_courses_this_month);

// Manejo de notificaciones de sesión (si existen)
$notification_message = '';
$notification_type = '';
if (isset($_SESSION['notification'])) {
    $notification_message = $_SESSION['notification']['message'];
    $notification_type = $_SESSION['notification']['type'];
    unset($_SESSION['notification']); // Limpiar la notificación después de mostrarla
}
ob_end_clean(); // Finalizar y limpiar el búfer de salida
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>LocalCRM ToolKit Dashboard | Gestión de Academia</title>
    <meta name="description" content="Administra tus cursos y contenido de la academia en LocalCRM: sube, edita y organiza tu material educativo para tus clientes.">
    <meta name="robots" content="noindex, nofollow">

    <link rel="icon" type="image/png" href="img/favicon.png">
    <link rel="apple-touch-icon" href="img/apple-touch-icon.png">

    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Barlow:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <script src="https://unpkg.com/lucide@latest"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    
    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script> 
    
    <style>
.toast-notification {
    visibility: hidden;
    min-width: 250px;
    background-color: #333;
    color: #fff;
    text-align: center;
    border-radius: 8px;
    padding: 16px;
    position: fixed;
    z-index: 1000;
    left: 50%;
    transform: translateX(-50%);
    bottom: 30px;
    font-size: 17px;
    display: flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
    opacity: 0;
    transition: opacity 0.5s, bottom 0.5s;
}

.toast-notification.show {
    visibility: visible;
    opacity: 1;
    bottom: 50px;
}

.toast-notification.success { background-color: #4CAF50; }
.toast-notification.error { background-color: #f44336; }
.toast-notification.info { background-color: #2196F3; }

.toast-notification .toast-close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 20px;
    cursor: pointer;
    margin-left: auto;
}
</style>

</head>
 
<body data-page-title="PANEL DE ADMINISTRADOR - ACADEMIA"
      data-page-subtitle="GESTIÓN DE CONTENIDO EDUCATIVO"
      data-page-icon="graduation-cap">

<div id="toastNotification" class="toast-notification">
    <i id="toastIcon" class="fas"></i>
    <span id="toastMessage"></span>
    <button id="toastCloseBtn" class="toast-close-btn">×</button>
</div>

<div class="relative min-h-screen md:flex">

    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>

    <?php include 'menu.php'; ?>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>

        <div id="content-area" class="p-4 md:p-8 space-y-8">
            <section id="academy-content" class="dashboard-section">
                <div class="bg-white p-6 rounded-xl shadow-md">
                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4">
                        <div>
                            <h3 class="text-2xl font-extrabold text-gray-800 flex items-center gap-2">
                                <i data-lucide="graduation-cap" class="w-7 h-7 text-[var(--color-primary)]"></i> INVENTARIO DE CURSOS
                            </h3>
                            <p class="text-gray-500 text-sm mt-1 uppercase">VISUALIZA Y ADMINISTRA TUS CURSOS DE LA ACADEMIA.</p>
                        </div>
                        <div class="flex items-center gap-4 w-full md:w-auto">
                            <button id="addCourseBtn" class="btn-secondary font-bold py-2 px-4 rounded-lg flex items-center w-full md:w-auto justify-center uppercase">
                                <i data-lucide="plus" class="w-5 h-5 mr-2"></i> AGREGAR CURSO
                            </button>
                            <button id="download-courses-btn" class="btn-primary font-bold py-2 px-4 rounded-lg flex items-center w-full md:w-auto justify-center uppercase text-sm">
                                <i data-lucide="download" class="w-4 h-4 mr-2"></i> DESCARGAR
                            </button>
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                        <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-primary)]">
                            <i data-lucide="book-open" class="w-12 h-12 text-gray-700"></i>
                            <div>
                                <h3 class="text-lg font-extrabold text-gray-500 mb-1">TOTAL DE CURSOS</h3>
                                <p id="total-courses-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
                                <p class="text-sm text-gray-400 mt-1">EN EL CATÁLOGO</p>
                            </div>
                        </div>
                        <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
                            <i data-lucide="calendar-check" class="w-12 h-12 text-gray-700"></i>
                            <div>
                                <h3 class="text-lg font-extrabold text-gray-500 mb-1">CURSOS NUEVOS ESTE MES</h3>
                                <p id="new-courses-this-month-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
                                <p class="text-sm text-gray-400 mt-1 uppercase">CURSOS AGREGADOS EN EL PERIODO</p>
                            </div>
                        </div>
                    </div>

                    <div class="flex flex-col md:flex-row gap-4 mb-6">
                        <div class="relative flex-grow">
                            <input type="text" id="course-search" placeholder="Buscar Por Título o Descripción..." class="w-full p-3 pl-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]">
                            <i data-lucide="search" class="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"></i>
                        </div>
                        <div class="view-toggle inline-flex bg-gray-200 rounded-lg p-1">
                            <button id="grid-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 active uppercase" title="Vista de Cuadrícula">
                                <i data-lucide="layout-grid" class="w-5 h-5"></i>
                            </button>
                            <button id="list-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 uppercase" title="Vista de Lista">
                                <i data-lucide="list" class="w-5 h-5"></i>
                            </button>
                        </div>
                    </div>

                    <div id="course-grid-view" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        </div>

                    <div id="course-list-view" class="overflow-x-auto hidden">
                        <table class="min-w-full bg-white">
                            <thead class="bg-gray-50">
                                <tr class="text-left text-gray-500 uppercase text-sm">
                                    <th class="py-3 px-6 font-semibold">TÍTULO</th>
                                    <th class="py-3 px-6 font-semibold hidden md:table-cell">DESCRIPCIÓN</th>
                                    <th class="py-3 px-6 font-semibold">URL</th>
                                    <th class="py-3 px-6 font-semibold text-center">ACCIONES</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-700 text-sm" id="coursesTableBody">
                                </tbody>
                        </table>
                    </div>

                </div>
            </section>
        </div>
    </main>
</div>

<div id="addCoursePanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">AGREGAR NUEVO CURSO</h3>
            <button onclick="closePanel('addCoursePanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="add-course-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <div class="grid grid-cols-1 gap-x-6 gap-y-4">
                <div>
                    <label for="course-titulo" class="block text-gray-700 text-sm font-bold mb-2 uppercase">TÍTULO DEL CURSO</label>
                    <input type="text" id="course-titulo" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. Introducción a CRM" required>
                </div>
                <div>
                    <label for="course-url" class="block text-gray-700 text-sm font-bold mb-2 uppercase">URL DEL CURSO (YOUTUBE/VIMEO/OTRO)</label>
                    <input type="url" id="course-url" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="https://www.youtube.com/watch?v=ejemplo" required>
                </div>
                <div>
                    <label for="course-poster-url" class="block text-gray-700 text-sm font-bold mb-2 uppercase">URL DE IMAGEN PÓSTER (OPCIONAL)</label>
                    <input type="url" id="course-poster-url" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. https://ejemplo.com/imagen_poster.jpg">
                    <p class="text-xs text-gray-500 mt-1">Si se especifica, esta imagen se usará como miniatura principal. De lo contrario, se generará desde la URL del curso.</p>
                </div>
                <div>
                    <label for="course-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="course-descripcion" rows="5" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Detallada Del Curso"></textarea>
                </div>
            </div>
            <div class="h-4"></div>
        </form>
        <div class="flex justify-end space-x-4 pt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('addCoursePanel')">CANCELAR</button>
            <button type="submit" form="add-course-form" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR CURSO</button>
        </div>
    </div>
</div>

<div id="confirmDeleteModal" class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center hidden z-50">
    <div class="bg-white p-8 rounded-xl shadow-2xl w-full max-w-sm m-4 transform transition-all duration-300 scale-95 opacity-0 text-center">
        <div class="flex justify-center mb-4">
            <i data-lucide="alert-triangle" class="w-16 h-16 text-red-500"></i>
        </div>
        <h3 class="text-2xl font-bold text-[var(--color-primary)] mb-4 uppercase">CONFIRMAR ELIMINACIÓN</h3>
        <p class="text-gray-700 mb-6 uppercase">¿ESTÁS SEGURO DE QUE DESEAS ELIMINAR ESTE <span id="confirm-item-type" class="font-semibold">ELEMENTO</span>? ESTA ACCIÓN NO SE PUEDE DESHACER.</p>
        <div class="flex justify-center space-x-4">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closeModal('confirmDeleteModal')">CANCELAR</button>
            <button type="button" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase" id="confirm-delete-button">CONFIRMAR</button>
        </div>
    </div>
</div>

<div id="viewCoursePanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">DETALLE DEL CURSO</h3>
            <button onclick="closePanel('viewCoursePanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="view-course-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <input type="hidden" id="view-course-id">
            <div class="grid grid-cols-1 gap-x-6 gap-y-4">
                <div>
                    <label for="view-course-titulo" class="block text-gray-700 text-sm font-bold mb-2 uppercase">TÍTULO DEL CURSO</label>
                    <input type="text" id="view-course-titulo" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Título Del Curso">
                </div>
                <div>
                    <label for="view-course-url" class="block text-gray-700 text-sm font-bold mb-2 uppercase">URL DEL CURSO (YOUTUBE/VIMEO/OTRO)</label>
                    <input type="url" id="view-course-url" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="https://www.youtube.com/watch?v=ejemplo">
                </div>
                <div>
                    <label for="view-course-poster-url" class="block text-gray-700 text-sm font-bold mb-2 uppercase">URL DE IMAGEN PÓSTER (OPCIONAL)</label>
                    <input type="url" id="view-course-poster-url" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. https://ejemplo.com/imagen_poster.jpg">
                    <p class="text-xs text-gray-500 mt-1">Si se especifica, esta imagen se usará como miniatura principal. De lo contrario, se generará desde la URL del curso.</p>
                </div>
                <div>
                    <label for="view-course-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="view-course-descripcion" rows="5" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Detallada Del Curso"></textarea>
                </div>
                <div class="md:col-span-2">
                    <label class="block text-gray-700 text-sm font-bold mb-2 uppercase">MINIATURA DEL CURSO</label>
                    <div id="course-thumbnail-preview" class="w-full bg-gray-200 aspect-video rounded-lg flex items-center justify-center text-gray-500">
                        <i data-lucide="image" class="w-12 h-12"></i>
                    </div>
                    <p class="text-xs text-gray-500 mt-2">La miniatura se genera automáticamente desde la URL del curso, o se utiliza la URL del póster si se proporciona.</p>
                </div>
            </div>
            <div class="h-4"></div>
        </form>
        <div class="flex justify-end space-x-4 pt-4 border-t border-gray-200 mt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('viewCoursePanel')">CANCELAR</button>
            <button type="submit" form="view-course-form" class="btn-primary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR CAMBIOS</button>
        </div>
    </div>
</div>

<script>
// Variables globales para los datos
let currentCourses = {}; // Se inicializa vacío, se llenará con fetchCourses()

document.addEventListener('DOMContentLoaded', function() {
    // --- CONSOLE.LOGS PARA DEPURACIÓN ---
    console.log("--- Depuración de academy.php ---");
    // --- FIN CONSOLE.LOGS PARA DEPURACIÓN ---

    // --- VARIABLES DOM LOCALES (Elementos HTML que el JS necesita manipular) ---
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebar = document.getElementById('sidebar');
    const sidebarOverlay = document.getElementById('sidebar-overlay');
    const sidebarCloseBtn = document.getElementById('sidebarCloseBtn');
    
    // Academia/Cursos
    const courseSearch = document.getElementById('course-search');
    const courseGridView = document.getElementById('course-grid-view');
    const courseListView = document.getElementById('course-list-view');
    const coursesTableBody = document.getElementById('coursesTableBody');
    const gridViewBtn = document.getElementById('grid-view-btn');
    const listViewBtn = document.getElementById('list-view-btn');
    const downloadCoursesBtn = document.getElementById('download-courses-btn');
    const addCourseForm = document.getElementById('add-course-form');
    const viewCourseForm = document.getElementById('view-course-form');
    const totalCoursesCountElement = document.getElementById('total-courses-count');
    const newCoursesThisMonthCountElement = document.getElementById('new-courses-this-month-count');
    const addCourseBtn = document.getElementById('addCourseBtn'); 

    // Referencias para el panel de ver/editar curso
    const viewCourseUrlInput = document.getElementById('view-course-url');
    // CAMBIO: Obtener referencia al nuevo campo de URL de póster en el formulario de edición
    const viewCoursePosterUrlInput = document.getElementById('view-course-poster-url');
    const courseThumbnailPreview = document.getElementById('course-thumbnail-preview');
    // CAMBIO: Obtener referencia al nuevo campo de URL de póster en el formulario de agregar
    const addCoursePosterUrlInput = document.getElementById('course-poster-url');


    // Modales de confirmación (compartidos)
    const confirmDeleteModal = document.getElementById('confirmDeleteModal');
    const confirmDeleteButton = document.getElementById('confirm-delete-button');
    const confirmItemTypeSpan = document.getElementById('confirm-item-type');


    // Variable para controlar la vista actual de los cursos (cuadrícula por defecto)
    let currentCourseView = 'grid'; 
    
    // Elemento de overlay para los paneles laterales (creado dinámicamente si no existe)
    let panelOverlay = document.getElementById('panel-overlay');
    if (!panelOverlay) {
        panelOverlay = document.createElement('div');
        panelOverlay.id = 'panel-overlay';
        panelOverlay.className = 'fixed inset-0 bg-black bg-opacity-50 z-40 hidden';
        document.body.appendChild(panelOverlay);
    }


    // --- FUNCIONES HELPER GLOBALES (acciones comunes que pueden ser llamadas desde cualquier parte) ---

    // Muestra una notificación temporal en la pantalla (como un "toast")
    window.showCustomNotification = function(message, type = 'info', duration = 3000) {
        const toast = document.getElementById('toastNotification');
        const toastMessage = document.getElementById('toastMessage');
        const toastIcon = document.getElementById('toastIcon');
        const toastCloseBtn = document.getElementById('toastCloseBtn');
        let toastTimeout;

        // Si los elementos del toast no existen, imprime un mensaje en consola y no hace nada
        if (!toast || !toastMessage || !toastIcon || !toastCloseBtn) {
            console.warn(`[NOTIFICACION - FALLBACK] Elementos de toast no encontrados. Mensaje: ${type.toUpperCase()}: ${message}`);
            return;
        }

        clearTimeout(toastTimeout); // Limpia cualquier notificación anterior
        // Limpiar clases de tipo anteriores para aplicar la nueva
        toast.classList.remove('success', 'error', 'info');
        toastIcon.classList.remove('fa-check-circle', 'fa-exclamation-circle', 'fa-info-circle');
        toast.className = 'toast-notification'; // Restablecer a la clase base

        toastMessage.textContent = message; // Establece el mensaje
        toast.classList.add(type); // Añade la clase de estilo (success, error, info)
        
        // Define el ícono basado en el tipo de notificación
        if (type === 'success') {
            toastIcon.classList.add('fa-check-circle');
        } else if (type === 'error') {
            toastIcon.classList.add('fa-exclamation-circle');
        } else {
            toastIcon.classList.add('fa-info-circle'); 
        }

        toast.classList.add('show'); // Muestra el toast
        // Configura un temporizador para ocultar el toast después de 'duration' milisegundos
        toastTimeout = setTimeout(() => {
            toast.classList.remove('show');
        }, duration);

        // Permite cerrar el toast manualmente
        toastCloseBtn.onclick = () => {
            clearTimeout(toastTimeout);
            toast.classList.remove('show');
        };

        console.log(`[NOTIFICACION - ${type.toUpperCase()}] ${message}`);
    };

    // Abre un panel lateral deslizante (como los de añadir/editar)
    window.openPanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { console.error(`Error: Panel con ID "${panelId}" no encontrado.`); return; }
        panel.classList.remove('translate-x-full'); // Mueve el panel para que sea visible
        panelOverlay.classList.remove('hidden');    // Muestra el overlay oscuro
        console.log(`[academy.php] Panel ${panelId} abierto.`);
        
        // Lógica específica para restablecer formularios al abrir
        if (panelId === 'addCoursePanel') {
            addCourseForm.reset(); // Limpia el formulario
            document.getElementById('course-url').value = ''; // Limpia la URL del input
            // CAMBIO: Limpia el nuevo campo de poster_url al abrir el panel de agregar
            if (addCoursePosterUrlInput) addCoursePosterUrlInput.value = '';
            // Restablece la previsualización de miniatura al icono por defecto
            const addCourseThumbnailPreview = document.getElementById('course-thumbnail-preview'); // Obtén el elemento de preview en el form de agregar
            if (addCourseThumbnailPreview) {
                addCourseThumbnailPreview.innerHTML = `<i data-lucide="image" class="w-12 h-12 text-gray-500"></i>`;
                lucide.createIcons({ container: addCourseThumbnailPreview });
            }
        }
    };

    // Cierra un panel lateral deslizante
    window.closePanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { console.error(`Error: Panel con ID "${panelId}" no encontrado para cerrar.`); return; }
        panel.classList.add('translate-x-full'); // Esconde el panel
        panelOverlay.classList.add('hidden');    // Esconde el overlay
        console.log(`[academy.php] Panel ${panelId} cerrado.`);
    };

    // Abre un modal (ventana emergente, como la de confirmación de eliminación)
    window.openModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { console.error(`Error: Modal con ID "${modalId}" no encontrado.`); return; }
        const modalBox = modal.querySelector('div:first-of-type'); // El contenido real del modal
        modal.classList.remove('hidden'); // Muestra el modal
        // Pequeño retardo para la animación de entrada
        setTimeout(() => { 
            if (modalBox) modalBox.classList.remove('scale-95', 'opacity-0'); 
        }, 50);
        console.log(`[academy.php] Modal ${modalId} abierto.`);
        lucide.createIcons({ container: modal }); // Asegura que los íconos del modal se muestren
    };

    // Cierra un modal
    window.closeModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { console.error(`Error: Modal con ID "${modalId}" no encontrado para cerrar.`); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        if (modalBox) modalBox.classList.add('scale-95', 'opacity-0'); // Inicia la animación de salida
        setTimeout(() => { modal.classList.add('hidden'); }, 300);
        console.log(`[academy.php] Modal ${modalId} cerrado.`);
    };
    
    // Prepara y abre el modal de confirmación de eliminación para un elemento específico
    window.openConfirmDeleteModal = function(itemId, itemType) {
        if (!confirmDeleteButton || !confirmItemTypeSpan) { console.error("Error: Elementos del modal de confirmación de eliminación no encontrados."); return; }
        confirmDeleteButton.dataset.itemId = itemId; // Guarda el ID del elemento a eliminar
        confirmDeleteButton.dataset.itemType = itemType; // Guarda el tipo (curso)
        confirmItemTypeSpan.textContent = itemType.toUpperCase(); // Muestra el tipo en el mensaje
        openModal('confirmDeleteModal'); // Abre el modal
    };

    // Descarga datos como CSV (adaptada para cursos)
    function descargarComoCSV(dataObject, filename) {
        const data = Object.values(dataObject);
        if (data.length === 0) { showCustomNotification('No hay datos para descargar.', 'info'); return; }
        const escapeCSV = (value) => {
            if (value === null || value === undefined) { return ''; }
            value = String(value);
            // Si el valor contiene comas, comillas o saltos de línea, lo encierra en comillas dobles y escapa las comillas dobles
            if (value.search(/("|,|\n)/g) >= 0) { return `"${value.replace(/"/g, '""')}"`; }
            return value;
        };
        // CAMBIO: Añadir 'poster_url' a los encabezados del CSV
        const headers = ['db_id', 'titulo', 'descripcion', 'url', 'poster_url', 'created_at']; 
        // Crea la primera fila del CSV con los encabezados
        const csvRows = [headers.map(h => escapeCSV(h.replace(/_/g, ' ').toUpperCase())).join(',')];
        // Añade cada fila de datos al CSV
        data.forEach(item => {
            const values = headers.map(header => escapeCSV(item[header]));
            csvRows.push(values.join(','));
        });
        const csvContent = csvRows.join('\n'); // Une todas las filas con saltos de línea
        // Crea un Blob para descargar el archivo
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a'); // Crea un enlace temporal
        if (link.download !== undefined) { // Asegura la compatibilidad con navegadores
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url); 
            link.setAttribute('download', filename); // Define el nombre del archivo
            link.style.visibility = 'hidden'; 
            document.body.appendChild(link); // Añade el enlace al DOM
            link.click(); // Simula un clic para descargar
            document.body.removeChild(link); // Elimina el enlace
            showCustomNotification('CSV generado y descargado.', 'success');
        } else {
            showCustomNotification('Tu navegador no soporta la descarga directa de archivos.', 'error');
        }
    }

    // CAMBIO: Función para extraer el ID de YouTube/Vimeo y obtener la URL de la miniatura, ahora prioriza posterUrl
    function getCourseThumbnail(url, posterUrl = null) {
        if (posterUrl && posterUrl.trim() !== '') {
            return posterUrl; // Si se proporciona una URL de póster, úsala
        }
        let thumbnailUrl = '';
        // Intenta extraer miniatura de YouTube
        const youtubeMatch = url.match(/(?:youtube\.com\/(?:watch\?v=|embed\/|v\/)|youtu\.be\/)([a-zA-Z0-9_-]{11})/);
        if (youtubeMatch && youtubeMatch[1]) {
            return `https://img.youtube.com/vi/${youtubeMatch[1]}/hqdefault.jpg`; 
        }

        // Intenta extraer miniatura de Vimeo
        const vimeoMatch = url.match(/(?:vimeo\.com\/)(\d+)/);
        if (vimeoMatch && vimeoMatch[1]) {
            return `https://vumbnail.com/${vimeoMatch[1]}.jpg`; 
        }
        
        // Si no es un formato reconocido o no se puede obtener miniatura, devuelve un placeholder
        return 'https://placehold.co/128x128/cccccc/333333?text=Curso'; 
    }


    // Abre el panel de Ver/Editar Curso
    window.openViewCourseModal = function(courseId) {
        const course = currentCourses[courseId];
        if (!course) { console.error('Curso no encontrado:', courseId); return; }
        
        // Rellena los campos del formulario con los datos del curso
        document.getElementById('view-course-id').value = course.db_id;
        document.getElementById('view-course-titulo').value = course.titulo;
        document.getElementById('view-course-url').value = course.url;
        // CAMBIO: Rellenar el nuevo campo de poster_url en el formulario de edición
        if (viewCoursePosterUrlInput) viewCoursePosterUrlInput.value = course.poster_url || '';
        document.getElementById('view-course-descripcion').value = course.descripcion || '';

        // Actualizar la miniatura del curso en el panel de edición
        // CAMBIO: Pasar el poster_url a getCourseThumbnail para que lo priorice
        const thumbnailUrl = getCourseThumbnail(course.url, course.poster_url);
        if (courseThumbnailPreview) {
            if (thumbnailUrl) {
                courseThumbnailPreview.innerHTML = `<img src="${thumbnailUrl}" alt="Miniatura de ${course.titulo}" class="w-full h-full object-cover rounded-lg">`;
            } else {
                courseThumbnailPreview.innerHTML = `<i data-lucide="image" class="w-12 h-12 text-gray-500"></i>`;
            }
            lucide.createIcons({ container: courseThumbnailPreview });
        }

        openPanel('viewCoursePanel');
    };

    // Setup de listeners para botones de eliminar
    function setupDeleteListeners() {
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.removeEventListener('click', handleDeleteButtonClick);
            button.addEventListener('click', handleDeleteButtonClick);
        });
    }
    function handleDeleteButtonClick(e) {
        e.stopPropagation();
        const itemId = this.dataset.itemId;
        const itemType = this.dataset.itemType;
        openConfirmDeleteModal(itemId, itemType);
    }

    // Setup de listeners para botones de ver/editar
    function setupEditListeners() {
        document.querySelectorAll('.edit-course-btn').forEach(button => {
            button.removeEventListener('click', handleEditCourseButtonClick);
            button.addEventListener('click', handleEditCourseButtonClick);
        });
    }
    function handleEditCourseButtonClick(e) {
        e.stopPropagation();
        const courseId = this.dataset.courseId;
        openViewCourseModal(courseId);
    }
    
    // Función para actualizar el conteo total de cursos en el dashboard
    function updateTotalCoursesDisplay() {
        if (totalCoursesCountElement) {
            const total = Object.keys(currentCourses).length;
            totalCoursesCountElement.textContent = total.toString();
        }
    }

    // Función para actualizar el contador de cursos nuevos este mes en el dashboard
    function updateNewCoursesThisMonthDisplay() {
        if (newCoursesThisMonthCountElement) {
            let newThisMonth = 0;
            const currentMonth = new Date().getMonth();
            const currentYear = new Date().getFullYear();
            Object.values(currentCourses).forEach(course => {
                if (course.created_at) {
                    const createdAt = new Date(course.created_at);
                    if (createdAt.getFullYear() === currentYear && createdAt.getMonth() === currentMonth) {
                        newThisMonth++;
                    }
                }
            });
            newCoursesThisMonthCountElement.textContent = newThisMonth.toString();
        }
    }

    // Crea el HTML para una tarjeta de curso (vista de cuadrícula)
    function renderCourseCard(course) {
        const card = document.createElement('div');
        card.className = 'bg-white rounded-xl shadow-md overflow-hidden flex flex-col border border-gray-200';
        card.dataset.courseId = course.id;
        card.dataset.itemId = course.id;

        // CAMBIO: Pasar el poster_url a getCourseThumbnail para que lo priorice
        const thumbnailUrl = getCourseThumbnail(course.url, course.poster_url);
        const thumbnailHtml = thumbnailUrl ? 
            `<img src="${thumbnailUrl}" alt="Miniatura de ${course.titulo}" class="w-full h-32 object-cover">` :
            `<div class="w-full h-32 bg-gray-200 flex items-center justify-center text-gray-500"><i data-lucide="image" class="w-12 h-12"></i></div>`;

        card.innerHTML = `
            <div class="relative">
                ${thumbnailHtml}
                <div class="absolute inset-0 flex items-center justify-center bg-black bg-opacity-50 text-white opacity-0 hover:opacity-100 transition-opacity duration-300 cursor-pointer" onclick="openViewCourseModal('${course.id}')">
                    <i data-lucide="play-circle" class="w-16 h-16"></i>
                </div>
            </div>
            <div class="p-4 flex-grow flex flex-col">
                <h3 class="font-bold text-lg text-gray-800 mb-2 uppercase line-clamp-2">${course.titulo}</h3>
                <p class="text-gray-700 text-sm flex-grow mb-4 line-clamp-3">${course.descripcion || 'Sin descripción detallada.'}</p>
                <p class="text-gray-500 text-xs mt-auto">Publicado: <span class="font-semibold">${new Date(course.created_at).toLocaleDateString()}</span></p>
            </div>
            <div class="bg-gray-100 p-3 flex justify-end space-x-3 border-t border-gray-200">
                <a href="${course.url}" target="_blank" class="bg-blue-100 hover:bg-blue-200 text-blue-600 font-semibold py-2 px-4 rounded-lg text-sm uppercase inline-flex items-center justify-center gap-1">
                    <i data-lucide="external-link" class="w-4 h-4"></i> Ir
                </a>
                <button class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-semibold py-2 px-4 rounded-lg text-sm uppercase edit-course-btn" data-course-id="${course.id}">EDITAR</button>
                <button class="p-2 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg delete-btn" data-item-type="course" data-item-id="${course.id}" title="ELIMINAR CURSO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
            </div>
        `;
        return card;
    }

    // Crea el HTML para una fila de tabla de curso (vista de lista)
    function renderCourseTableRow(course) {
        const row = document.createElement('tr');
        row.className = 'border-b border-gray-200 hover:bg-gray-50';
        row.dataset.courseId = course.id;
        row.dataset.itemId = course.id;

        const shortDescription = course.descripcion ? course.descripcion.substring(0, 70) + (course.descripcion.length > 70 ? '...' : '') : 'N/A';

        row.innerHTML = `
            <td class="py-4 px-6 whitespace-nowrap">
                <span class="text-blue-600 hover:text-blue-800 cursor-pointer font-semibold uppercase" onclick="openViewCourseModal('${course.id}')">${course.titulo}</span>
            </td>
            <td class="py-4 px-6 hidden md:table-cell uppercase">${shortDescription}</td>
            <td class="py-4 px-6">
                <a href="${course.url}" target="_blank" class="text-blue-500 hover:underline flex items-center gap-1">
                    <i data-lucide="external-link" class="w-4 h-4"></i> Enlace
                </a>
            </td>
            <td class="py-4 px-6 text-center">
                <button class="text-blue-600 hover:text-blue-800 mr-3 edit-course-btn" title="EDITAR CURSO" data-course-id="${course.id}"><i data-lucide="edit-2" class="w-5 h-5"></i></button>
                <button class="text-red-600 hover:text-red-800 delete-btn" data-item-type="course" data-item-id="${course.id}" title="ELIMINAR CURSO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
            </td>
        `;
        return row;
    }

    // Renderiza (dibuja) todos los cursos en la vista seleccionada (cuadrícula o lista)
    function renderCourses(viewType = currentCourseView) {
        if (!courseGridView || !courseListView || !coursesTableBody || !courseSearch || !gridViewBtn || !listViewBtn) {
               console.error("[academy.php] ERROR: Elementos DOM de Cursos no encontrados. Verifique los IDs de los elementos HTML."); return;
        }
        const searchTerm = courseSearch.value.toLowerCase();

        const allCoursesArray = Object.values(currentCourses);

        console.log("renderCourses: currentCourses antes de filtrar:", allCoursesArray);

        const filteredCourses = allCoursesArray.filter(c => {
            const matchesSearch = searchTerm === '' || 
                                    (c.titulo && c.titulo.toLowerCase().includes(searchTerm)) || 
                                    (c.descripcion && c.descripcion.toLowerCase().includes(searchTerm)) || 
                                    (c.url && c.url.toLowerCase().includes(searchTerm));
            return matchesSearch;
        });

        console.log("renderCourses: cursos filtrados:", filteredCourses);


        courseGridView.innerHTML = '';
        coursesTableBody.innerHTML = '';

        if (filteredCourses.length === 0) {
            const message = `<p class="text-gray-500 md:col-span-2 lg:col-span-3 text-center uppercase">NO SE ENCONTRARON CURSOS.</p>`;
            if (viewType === 'grid') {
                courseGridView.innerHTML = message;
            } else {
                coursesTableBody.innerHTML = `<tr><td colspan="4" class="py-4 px-6 text-center text-gray-500 uppercase">NO SE ENCONTRARON CURSOS.</td></tr>`;
            }
        } else {
            if (viewType === 'grid') {
                filteredCourses.forEach(course => courseGridView.appendChild(renderCourseCard(course)));
                lucide.createIcons({ container: courseGridView });
            } else {
                filteredCourses.forEach(course => coursesTableBody.appendChild(renderCourseTableRow(course)));
                lucide.createIcons({ container: coursesTableBody });
            }
        }

        setupDeleteListeners();
        setupEditListeners();
    }
    
    function formatPhoneNumber(value) {
        if (!value) return '';
        const cleaned = ('' + value).replace(/\D/g, '');    
        const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
        if (match) {
            return `(${match[1]}) ${match[2]}-${match[3]}`;
        }
        return cleaned;      
    }

    function applyInputPhoneFormat(inputElement) {
        if (!inputElement) return;
        inputElement.value = formatPhoneNumber(inputElement.value);
        inputElement.addEventListener('input', (e) => {
            let value = e.target.value.replace(/\D/g, '');    
            e.target.value = formatPhoneNumber(value);
        });
        inputElement.addEventListener('blur', (e) => {
            e.target.value = formatPhoneNumber(e.target.value);
        });
    }

    function applyPhoneFormatToInputs() {
        // No hay campos de teléfono en esta página de academia.
    }


    // --- INICIALIZACIÓN DE LA PÁGINA ---

    // 1. Mostrar notificación PHP si existe
    const phpNotificationMessage = "<?php echo addslashes($notification_message); ?>";
    const phpNotificationType = "<?php echo addslashes($notification_type); ?>";
    if (phpNotificationMessage) {
        showCustomNotification(phpNotificationMessage, phpNotificationType);
    }

    // 2. Configurar eventos de la barra lateral (menú móvil)
    if (mobileMenuButton) {
        mobileMenuButton.addEventListener('click', typeof toggleSidebar === 'function' ? toggleSidebar : () => console.error('toggleSidebar function not found'));
    }

    // Cierra paneles y overlay si se hace clic en el overlay del sidebar (para móviles)
    if (sidebarOverlay) {
        sidebarOverlay.addEventListener('click', () => {
            if (typeof toggleSidebar === 'function') toggleSidebar();
            closePanel('addCoursePanel'); 
            closePanel('viewCoursePanel');
        });
    }
    if (panelOverlay) {
        panelOverlay.addEventListener('click', () => {   
            closePanel('addCoursePanel');
            closePanel('viewCoursePanel');
        });
    }
    if (sidebarCloseBtn) {
        if (typeof toggleSidebar === 'function') sidebarCloseBtn.addEventListener('click', toggleSidebar);
    }
    
    if (typeof window.toggleSidebar === 'undefined') {
        window.toggleSidebar = function() {
            const sidebarElement = document.getElementById('sidebar');
            const sidebarOverlayElement = document.getElementById('sidebar-overlay');
            if (sidebarElement && sidebarOverlayElement) {
                sidebarElement.classList.toggle('-translate-x-full');
                sidebarOverlayElement.classList.toggle('hidden');
            }
        };
    }


    // --- ACADEMIA: Lógica de inicialización y eventos ---
    const fetchCourses = async () => {
        try {
            const response = await fetch('db/academy-read.php?' + new Date().getTime());
            const result = await response.json();

            if (result.success && result.data) {
                currentCourses = result.data;
                console.log("Cursos cargados desde DB:", currentCourses);
                updateTotalCoursesDisplay();
                updateNewCoursesThisMonthDisplay();
                renderCourses(currentCourseView);
            } else {
                showCustomNotification(result.message || 'Error al cargar los cursos desde la base de datos.', 'error');
                currentCourses = {};
                updateTotalCoursesDisplay();
                updateNewCoursesThisMonthDisplay();
                renderCourses(currentCourseView);
            }
        } catch (error) {
            showCustomNotification('Error de conexión al cargar los cursos.', 'error');
            console.error('Error fetching courses:', error);
            currentCourses = {};
            updateTotalCoursesDisplay();
            updateNewCoursesThisMonthDisplay();
            renderCourses(currentCourseView);
        }
    };
    fetchCourses();
    console.log("[academy.php] Iniciando carga y renderizado de cursos.");

    // Evento de búsqueda para cursos
    if (courseSearch) courseSearch.addEventListener('input', () => renderCourses(currentCourseView));

    // Lógica para el botón de descarga CSV
    if(downloadCoursesBtn) {
        downloadCoursesBtn.addEventListener('click', () => {
            const filteredCourses = Object.values(currentCourses).filter(c => {
                const searchTerm = courseSearch.value.toLowerCase();
                return searchTerm === '' || 
                       (c.titulo && c.titulo.toLowerCase().includes(searchTerm)) || 
                       (c.descripcion && c.descripcion.toLowerCase().includes(searchTerm)) || 
                       (c.url && c.url.toLowerCase().includes(searchTerm));
            });
            descargarComoCSV(filteredCourses, 'catalogo_cursos.csv');
        });
    }
    
    // Lógica para el botón de confirmación de eliminación (compartido para cursos)
    if (confirmDeleteButton) {
        confirmDeleteButton.addEventListener('click', async function() {
            const itemId = this.dataset.itemId;
            const itemType = this.dataset.itemType;
            
            if (itemType === 'course') {
                try {
                    const response = await fetch('db/academy-delete.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ id: currentCourses[itemId].db_id })
                    });
                    const result = await response.json();

                    if (result.success) {
                        closeModal('confirmDeleteModal');
                        showCustomNotification(result.message, 'success');
                        fetchCourses();
                    } else {
                        showCustomNotification(result.message || 'Error al eliminar curso.', 'error');
                    }
                } catch (error) {
                    showCustomNotification('Error de conexión al eliminar el curso.', 'error');
                    console.error('Delete course error:', error);
                }
            }
        });
    }

    // Manejo de clics para los botones de vista (cuadrícula/lista) para cursos
    if (gridViewBtn) {
        gridViewBtn.addEventListener('click', () => {
            currentCourseView = 'grid';
            gridViewBtn.classList.add('active');
            listViewBtn.classList.remove('active');
            courseGridView.classList.remove('hidden');
            courseListView.classList.add('hidden');
            renderCourses('grid');
        });
    }
    if (listViewBtn) {
        listViewBtn.addEventListener('click', () => {
            currentCourseView = 'list';
            listViewBtn.classList.add('active');
            gridViewBtn.classList.remove('active');
            courseListView.classList.remove('hidden');
            courseGridView.classList.add('hidden');
            renderCourses('list');
        });
    }

    // Lógica para el formulario de añadir nuevos cursos
    if (addCourseForm) {
        addCourseForm.addEventListener('submit', async function(e) {   
            e.preventDefault();
            const courseData = {
                titulo: document.getElementById('course-titulo').value,
                url: document.getElementById('course-url').value,
                // CAMBIO: Incluir el nuevo campo poster_url en los datos enviados
                poster_url: addCoursePosterUrlInput ? addCoursePosterUrlInput.value : '',
                descripcion: document.getElementById('course-descripcion').value
            };

            try {   
                const response = await fetch('db/academy-create.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(courseData)
                });
                const result = await response.json();

                if (result.success) {
                    closePanel('addCoursePanel');
                    addCourseForm.reset();
                    showCustomNotification(result.message, 'success');
                    fetchCourses();
                } else {
                    showCustomNotification(result.message || 'Error al agregar curso.', 'error');
                }
            } catch (error) {
                showCustomNotification('Error de conexión al agregar el curso.', 'error');
                console.error('Create course error:', error);
            }
        });
    }

    // Lógica para el formulario de ver/editar cursos
    if (viewCourseForm) {
        viewCourseForm.addEventListener('submit', async function(e) {   
            e.preventDefault();
            const courseData = {
                id: document.getElementById('view-course-id').value,
                titulo: document.getElementById('view-course-titulo').value,
                url: document.getElementById('view-course-url').value,
                // CAMBIO: Incluir el nuevo campo poster_url en los datos enviados al actualizar
                poster_url: viewCoursePosterUrlInput ? viewCoursePosterUrlInput.value : '',
                descripcion: document.getElementById('view-course-descripcion').value
            };
            
            try {   
                const response = await fetch('db/academy-update.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(courseData)
                });
                const result = await response.json();

                if (result.success) {
                    closePanel('viewCoursePanel');
                    showCustomNotification(result.message, 'success');
                    fetchCourses();
                } else {
                    showCustomNotification(result.message || 'Error al guardar cambios.', 'error');
                }
            } catch (error) {
                showCustomNotification('Error de conexión al guardar los cambios.', 'error');
                console.error('Update course error:', error);
            }
        });
    }

    // CAMBIO: Listener para actualizar la miniatura en el panel de edición cuando cambia la URL del curso O la URL del póster
    if (viewCourseUrlInput && viewCoursePosterUrlInput && courseThumbnailPreview) {
        const updateThumbnailDisplay = () => {
            const videoUrl = viewCourseUrlInput.value;
            const posterUrl = viewCoursePosterUrlInput.value;
            // CAMBIO: Llama a getCourseThumbnail con ambos URLs para que decida cuál usar
            const thumbnailUrl = getCourseThumbnail(videoUrl, posterUrl); 
            if (thumbnailUrl) {
                courseThumbnailPreview.innerHTML = `<img src="${thumbnailUrl}" alt="Miniatura de curso" class="w-full h-full object-cover rounded-lg">`;
            } else {
                courseThumbnailPreview.innerHTML = `<i data-lucide="image" class="w-12 h-12 text-gray-500"></i>`;
            }
            lucide.createIcons({ container: courseThumbnailPreview });
        };

        viewCourseUrlInput.addEventListener('input', updateThumbnailDisplay);
        viewCoursePosterUrlInput.addEventListener('input', updateThumbnailDisplay);
    }
    
    // Listener para el botón "Agregar Curso"
    if (addCourseBtn) {
        addCourseBtn.addEventListener('click', () => {
            openPanel('addCoursePanel');
        });
    }

    // Renderiza todos los íconos de Lucide estáticos al final de la carga del DOM
    lucide.createIcons();
});
</script>
</body>
</html>