<?php
// db/clients-create.php
ob_start(); // Iniciar el búfer de salida
error_reporting(0); // Desactivar la muestra de errores en producción
ini_set('display_errors', 0); // Desactivar la muestra de errores en producción

include_once 'db_connection.php'; // Incluir el archivo de conexión a la base de datos

$response = ['success' => false, 'message' => '', 'client' => null];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error de JSON en clients-create.php: " . json_last_error_msg());
        $response['message'] = 'Error al decodificar datos de solicitud.';
        echo json_encode($response);
        exit();
    }
    if (empty($input)) {
        $response['message'] = 'Datos de solicitud vacíos o no válidos.';
        echo json_encode($response);
        exit();
    }

    $first_name = trim($input['first_name'] ?? '');
    $last_name = trim($input['last_name'] ?? '');
    $email = filter_var(trim($input['email'] ?? ''), FILTER_SANITIZE_EMAIL) ?: null;
    $phone = trim($input['phone'] ?? '') ?: null;
    $mobile = trim($input['mobile'] ?? '') ?: null;
    // El campo 'city' se corresponde con 'state_province' en la base de datos según tu esquema
    $city = trim($input['ciudad'] ?? '') ?: null; 
    $state_province = trim($input['estado'] ?? '') ?: null; 
    $assigned_service_ids = $input['assigned_service_ids'] ?? []; // Array de IDs de servicios

    if (empty($first_name) || empty($last_name)) {
        $response['message'] = 'Nombre y apellido son obligatorios.';
        echo json_encode($response);
        exit();
    }

    if ($email && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $response['message'] = 'El formato del correo electrónico no es válido.';
        echo json_encode($response);
        exit();
    }

    try {
        $pdo->beginTransaction();

        // 1. Insertar el nuevo cliente
        $stmt_client = $pdo->prepare("INSERT INTO clients (first_name, last_name, email, phone, mobile, state_province) VALUES (:first_name, :last_name, :email, :phone, :mobile, :state_province)");
        $stmt_client->bindParam(':first_name', $first_name);
        $stmt_client->bindParam(':last_name', $last_name);
        $stmt_client->bindParam(':email', $email);
        $stmt_client->bindParam(':phone', $phone);
        $stmt_client->bindParam(':mobile', $mobile);
        $stmt_client->bindParam(':state_province', $state_province); // Usar state_province como campo de estado
        $stmt_client->execute();

        $new_client_id = $pdo->lastInsertId();

        // 2. Insertar los servicios asignados si los hay
        if (!empty($assigned_service_ids)) {
            $stmt_client_service = $pdo->prepare("INSERT INTO client_services (client_id, service_id) VALUES (:client_id, :service_id)");
            foreach ($assigned_service_ids as $service_id) {
                // Asegurarse de que el service_id sea un entero válido
                $service_id_int = (int)$service_id;
                if ($service_id_int > 0) { // Validar que sea un ID de servicio positivo
                    $stmt_client_service->bindParam(':client_id', $new_client_id, PDO::PARAM_INT);
                    $stmt_client_service->bindParam(':service_id', $service_id_int, PDO::PARAM_INT);
                    $stmt_client_service->execute();
                }
            }
        }

        $pdo->commit();

        $response['success'] = true;
        $response['message'] = 'Cliente agregado con éxito.';
        $response['client'] = [
            'id' => $new_client_id,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            'phone' => $phone,
            'mobile' => $mobile,
            'city' => $city, // Mantenemos para la respuesta JS
            'estado' => $state_province, // Mantenemos para la respuesta JS
            'assigned_service_ids' => $assigned_service_ids,
            'created_at' => date('Y-m-d H:i:s') // Simular fecha de creación
        ];

    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error PDO al crear cliente: " . $e->getMessage());
        $response['message'] = 'Error de Base de Datos al crear cliente: ' . $e->getMessage();
        // Si el error es por duplicado de email:
        if ($e->getCode() == 23000) { // Código de error SQL para violación de integridad (ej. UNIQUE constraint)
            $response['message'] = 'El correo electrónico ya está registrado para otro cliente.';
        }
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error general al crear cliente: " . $e->getMessage());
        $response['message'] = 'Error inesperado al crear cliente: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Método de solicitud no válido.';
}

ob_end_clean(); // Limpiar el búfer de salida
header('Content-Type: application/json'); // Establecer el tipo de contenido a JSON
echo json_encode($response); // Enviar la respuesta JSON
exit; // Terminar la ejecución del script