// header-manager.js

// Definición global de la función showToast.
// No está dentro de DOMContentLoaded para que esté disponible inmediatamente para otros scripts.
/**
 * Muestra una notificación "toast" en la esquina superior derecha.
 * @param {string} message - El mensaje a mostrar en la notificación.
 * @param {string} type - El tipo de notificación ('success', 'error', 'info', 'warning').
 */
function showToast(message, type = 'info') { // Default type to 'info'
    const toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        console.error("Error: Contenedor de toast (#toast-container) no encontrado en el DOM. Asegúrate de que el HTML está presente y cargado en tu página.");
        return;
    }

    const toast = document.createElement('div');
    toast.classList.add('toast', `toast-${type}`);
    toast.innerHTML = `
        <div class="toast-icon">
            ${type === 'success' ? '<i class="fas fa-check-circle"></i>' : ''}
            ${type === 'error' ? '<i class="fas fa-times-circle"></i>' : ''}
            ${type === 'info' ? '<i class="fas fa-info-circle"></i>' : ''}
            ${type === 'warning' ? '<i class="fas fa-exclamation-triangle"></i>' : ''}
        </div>
        <div class="toast-message">${message}</div>
        <button class="toast-close" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    toastContainer.appendChild(toast);

    // Ocultar automáticamente después de 5 segundos
    setTimeout(() => {
        toast.classList.add('hide'); // Inicia la animación de ocultar (CSS)
        toast.addEventListener('transitionend', () => toast.remove()); // Elimina el elemento del DOM al finalizar la transición
    }, 5000); // 5000 ms = 5 segundos
}

// Hacer la función showToast global para fácil acceso desde otros scripts
window.showToast = showToast;
// También hacerla global como showCustomNotification para compatibilidad, ambas apuntan a la misma función.
window.showCustomNotification = showToast;


// Lógica de DOMContentLoaded para la CABECERA, NO para las notificaciones de URL.
document.addEventListener('DOMContentLoaded', function() {
    // 1. El script busca los elementos de la cabecera
    const pageTitleEl = document.getElementById('page-title');
    const pageSubtitleEl = document.getElementById('page-subtitle');
    const body = document.body;

    if (pageTitleEl && pageSubtitleEl && body) {
        // 2. Lee las "etiquetas de datos" del tag <body>
        const titleText = body.dataset.pageTitle || 'Dashboard';
        const subtitleText = body.dataset.pageSubtitle || 'Bienvenido';
        const iconName = body.dataset.pageIcon || 'home';

        // 3. Modificación: La función para dar formato al titulo, ahora con lógica condicional para móviles.
        function formatBicolorTitle(fullTitle, icon) {
            const isMobileScreen = window.innerWidth < 768; 
            if (isMobileScreen) {
                return `<i data-lucide="${icon}"></i><span class="main-title-part-1">${fullTitle.toUpperCase()}</span>`;
            } else {
                const words = fullTitle.split(' ');
                const middle = Math.ceil(words.length / 2);
                const firstPart = words.slice(0, middle).join(' ');
                const secondPart = words.slice(middle).join(' ');
                return `<i data-lucide="${icon}"></i><span class="main-title-part-1">${firstPart.toUpperCase()}</span><span class="main-title-part-2">${secondPart.toUpperCase()}</span>`;
            }
        }

        // 4. Llenar la cabecera con la informacion
        pageTitleEl.innerHTML = formatBicolorTitle(titleText, iconName);
        pageSubtitleEl.textContent = subtitleText.toUpperCase();

        // 5. Activar el icono de Lucide
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

}); // Fin de document.addEventListener('DOMContentLoaded', ...);