<?php
header('Content-Type: application/json');

// Incluye el archivo de conexión a la base de datos
require_once 'db_connection.php';

$response = ['status' => 'error', 'message' => 'Solicitud no válida.'];

if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    // PHP no parseo automáticamente el input PUT/DELETE, lo leemos manualmente
    // La lectura de `file_get_contents('php://input')` ya devuelve el cuerpo de la solicitud tal cual.
    // Si JavaScript envía JSON, json_decode es el método correcto.
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        echo json_encode(['status' => 'error', 'message' => 'Error al decodificar JSON: ' . json_last_error_msg()]);
        exit;
    }

    $id = $data['id'] ?? null; // ID numérico sin prefijo
    $type = $data['type'] ?? null; // 'unica' o 'recurrente'
    $new_status = $data['status'] ?? null; // Usado para cambiar el estado de la cita

    // Validar ID y tipo
    if (!is_numeric($id) || empty($type)) {
        $response['message'] = 'Faltan datos obligatorios (id, type) o son inválidos.';
        echo json_encode($response);
        exit;
    }

    try {
        // --- Manejar actualización de estado (si se envía el campo 'status') ---
        if ($new_status) {
            if ($type === 'unica') {
                $stmt = $pdo->prepare("UPDATE unique_appointments SET status = ? WHERE id = ?");
                $stmt->execute([$new_status, $id]);
                $response = ['status' => 'success', 'message' => 'Estado de cita única actualizado con éxito.'];
            } elseif ($type === 'recurrente') {
                // Las citas recurrentes no tienen un campo 'status' en la tabla.
                // Esta acción no se guarda en la base de datos para citas recurrentes.
                // Si la lógica de negocio requiere el seguimiento de instancias completadas/canceladas de citas recurrentes,
                // se necesitaría una tabla intermedia (ej. `recurring_appointment_instances`)
                // o un cambio en el esquema de `recurring_appointments` para un estado global.
                // Por ahora, se simula el éxito para que el frontend actualice su estado local.
                $response = ['status' => 'success', 'message' => 'Operación de estado para cita recurrente reconocida, pero no aplicable a la base de datos directamente.'];
            } else {
                $response['message'] = 'Tipo de cita no válido para actualización de estado.';
            }
            echo json_encode($response); // Envía respuesta y sale después de actualizar el estado (o simularlo)
            exit;
        }

        // --- Manejar actualización de otros campos (edición general) ---
        // ✅ MODIFICADO: Ya no se espera service_id para actualizar, solo notas y/o campos específicos de fecha/frecuencia
        $notes = $data['notes'] ?? '';
        
        // No validamos lead_id aquí, ya que no es un campo que se edite directamente desde el panel de actualización.
        // Se asume que el lead_id es fijo una vez que la cita es creada.
        // Si se necesita la edición de lead_id, se requeriría una lógica de autocompletado en el panel de edición.

        if ($type === 'unica') {
            $appointment_date = $data['appointment_date'] ?? null;
            $appointment_time = $data['appointment_time'] ?? null;

            if (empty($appointment_date) || empty($appointment_time)) {
                $response['message'] = 'Faltan datos obligatorios para actualizar cita única (appointment_date, appointment_time).';
                echo json_encode($response);
                exit;
            }

            // ✅ MODIFICADO: service_id es reemplazado por lead_id, pero lead_id NO se actualiza aquí.
            // Solo se actualizan appointment_date, appointment_time y notes.
            $stmt = $pdo->prepare("UPDATE unique_appointments SET appointment_date = ?, appointment_time = ?, notes = ? WHERE id = ?");
            $stmt->execute([$appointment_date, $appointment_time, $notes, $id]);
            $response = ['status' => 'success', 'message' => 'Cita única actualizada con éxito.'];

        } elseif ($type === 'recurrente') {
            $frequency = $data['frequency'] ?? null;
            // 'next_appointment_date' no se actualiza directamente aquí, es un campo calculado o una referencia.
            // Si necesitaras cambiarlo, tendrías que re-calcularlo o permitir su edición.
            // Por ahora, solo se actualizan frequency y notes.

            if (empty($frequency)) {
                $response['message'] = 'Faltan datos obligatorios para actualizar cita recurrente (frequency).';
                echo json_encode($response);
                exit;
            }

            // ✅ MODIFICADO: service_id es reemplazado por lead_id, pero lead_id NO se actualiza aquí.
            // Solo se actualizan frequency y notes.
            $stmt = $pdo->prepare("UPDATE recurring_appointments SET frequency = ?, notes = ? WHERE id = ?");
            $stmt->execute([$frequency, $notes, $id]);
            $response = ['status' => 'success', 'message' => 'Cita recurrente actualizada con éxito.'];

        } else {
            $response['message'] = 'Tipo de cita no reconocido para actualización de campos.';
        }

    } catch (PDOException $e) {
        $response['message'] = 'Error de base de datos al actualizar cita: ' . $e->getMessage();
        error_log("Error al actualizar cita: " . $e->getMessage()); // Para depuración
    } catch (Exception $e) {
        $response['message'] = $e->getMessage();
        error_log("Error general al actualizar cita: " . $e->getMessage()); // Para depuración
    }
}

echo json_encode($response);
?>