<?php
// crm/db/convert-quote-to-invoice.php
// Script para convertir una cotización en una factura.

ob_start();
error_reporting(E_ALL); // Habilitado para depuración - CAMBIAR A 0 EN PRODUCCIÓN FINAL
ini_set('display_errors', 1); // Habilitado para depuración - CAMBIAR A 0 EN PRODUCCIÓN FINAL

include_once 'db_connection.php'; // Asegúrate de que esta ruta es correcta

$response = ['success' => false, 'message' => ''];

try {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error: JSON inválido recibido.');
    }

    $quote_db_id = filter_var($data['quote_id'] ?? null, FILTER_VALIDATE_INT);

    if ($quote_db_id === false) {
        throw new Exception('ID de cotización inválido.');
    }

    $pdo->beginTransaction();

    // 1. Obtener los detalles de la cotización y sus ítems
    $stmt_quote = $pdo->prepare("SELECT * FROM estimates WHERE id = ?");
    $stmt_quote->execute([$quote_db_id]);
    $quote = $stmt_quote->fetch(PDO::FETCH_ASSOC);

    if (!$quote) {
        throw new Exception('Cotización no encontrada.');
    }

    $stmt_quote_items = $pdo->prepare("SELECT description, unit_price, quantity FROM estimate_items WHERE estimate_id = ?");
    $stmt_quote_items->execute([$quote_db_id]);
    $quote_items = $stmt_quote_items->fetchAll(PDO::FETCH_ASSOC);

    if (empty($quote_items)) {
        throw new Exception('La cotización no tiene ítems para convertir.');
    }

    // --- INICIO DE LA LÓGICA AGREGADA PARA CONVERTIR PROSPECTO A CLIENTE SI ES NECESARIO ---
    $final_client_id = $quote['client_id']; // Por defecto, usa el client_id existente
    $final_lead_id = $quote['lead_id'];     // Por defecto, usa el lead_id existente

    // Si la cotización era para un lead (prospecto) y no para un cliente
    if ($quote['lead_id'] !== null && $quote['client_id'] === null) {
        // Obtener detalles del lead
        $stmt_lead = $pdo->prepare("SELECT * FROM leads WHERE id = ?");
        $stmt_lead->execute([$quote['lead_id']]);
        $lead = $stmt_lead->fetch(PDO::FETCH_ASSOC);

        if ($lead) {
            // Insertar detalles del lead en la tabla de clientes
            $stmt_insert_client = $pdo->prepare(
                "INSERT INTO clients (first_name, last_name, email, phone, mobile, street_address, city, state_province, zip_code)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
            );
            $stmt_insert_client->execute([
                $lead['first_name'],
                $lead['last_name'],
                $lead['email'],
                $lead['phone'],
                $lead['mobile'],
                $lead['street_address'],
                $lead['city'],
                $lead['state_province'],
                $lead['zip_code']
            ]);
            $new_client_id = $pdo->lastInsertId(); // Obtener el ID del cliente recién creado

            // Actualizar el lead a 'Convertido a Cliente' y marcarlo como no editable
            $stmt_update_lead = $pdo->prepare(
                "UPDATE leads SET status = 'Convertido a Cliente', is_uneditable = 1 WHERE id = ?"
            );
            $stmt_update_lead->execute([$lead['id']]);

            // Asignar el nuevo client_id para la factura y anular el lead_id
            $final_client_id = $new_client_id;
            $final_lead_id = null; // La factura ahora estará asociada al cliente, no al lead
        } else {
            throw new Exception('Lead asociado a la cotización no encontrado o ya no existe.');
        }
    }
    // --- FIN DE LA LÓGICA AGREGADA ---

    // 2. Insertar nueva factura en la tabla `invoices`
    $new_invoice_external_id = 'fac-' . uniqid(); // Generar un ID externo único
    $invoice_date = date('Y-m-d'); // Fecha actual para la factura
    $due_date = date('Y-m-d', strtotime($invoice_date . ' +7 days')); // Fecha de vencimiento, por ejemplo, 7 días después

    $stmt_insert_invoice = $pdo->prepare(
        "INSERT INTO invoices (external_id, client_id, lead_id, invoice_date, due_date, status, tax_rate, subtotal, total_amount, converted_from_estimate_id)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
    );

    $stmt_insert_invoice->execute([
        $new_invoice_external_id,
        $final_client_id, // Usar el client_id final (ya sea el original o el recién creado)
        $final_lead_id,   // Usar el lead_id final (será null si se convirtió a cliente)
        $invoice_date,
        $due_date,
        'generada', // Estado inicial de la nueva factura
        $quote['tax_rate'],
        $quote['subtotal'],
        $quote['total_amount'],
        $quote_db_id // Vincular a la cotización original
    ]);
    $new_invoice_db_id = $pdo->lastInsertId();

    // 3. Insertar los ítems de la factura en `invoice_items`
    $stmt_insert_item = $pdo->prepare(
        "INSERT INTO invoice_items (invoice_id, description, unit_price, quantity)
         VALUES (?, ?, ?, ?)"
    );

    foreach ($quote_items as $item) {
        $stmt_insert_item->execute([
            $new_invoice_db_id,
            $item['description'],
            $item['unit_price'],
            $item['quantity']
        ]);
    }

    // 4. Actualizar el estado de la cotización original y vincularla a la factura creada
    $stmt_update_quote = $pdo->prepare(
        "UPDATE estimates SET status = ?, converted_to_invoice_id = ? WHERE id = ?"
    );
    $stmt_update_quote->execute(['convertida', $new_invoice_db_id, $quote_db_id]);

    $pdo->commit();

    $response['success'] = true;
    $response['message'] = 'Cotización convertida a factura con éxito.';
    $response['invoice_external_id'] = $new_invoice_external_id;

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("PDO Error al convertir cotización a factura: " . $e->getMessage());
    $response['message'] = 'Error de Base de Datos: ' . $e->getMessage();
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("Error al convertir cotización a factura: " . $e->getMessage());
    $response['message'] = $e->getMessage();
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>