<?php
// db/quote-create.php
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

include_once 'db_connection.php';

$response = [];

try {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error: JSON inválido recibido. ' . json_last_error_msg());
    }

    // Validar datos requeridos para el estimado principal
    if (empty($data['estimate_date']) || empty($data['status']) || (!isset($data['client_id']) && !isset($data['lead_id'])) || !isset($data['items'])) {
        throw new Exception('Datos incompletos para el estimado. Fecha, estado, contacto e ítems son requeridos.');
    }
    if (!is_array($data['items']) || empty($data['items'])) {
        throw new Exception('Los ítems del estimado deben ser un array no vacío.');
    }

    $client_id = !empty($data['client_id']) ? filter_var($data['client_id'], FILTER_VALIDATE_INT) : null;
    $lead_id = !empty($data['lead_id']) ? trim($data['lead_id']) : null;
    $estimate_date = $data['estimate_date'];
    // IMPORTANTE: El estado inicial ahora se establece aquí como 'generado'
    $status = trim($data['status']); // Aunque se recibe, lo forzaremos a 'generado' si es un nuevo estimado o lo dejamos si es 'generado'
    if (empty($data['id'])) { // Si es una creación (no una actualización)
        $status = 'generado'; // Forzar estado inicial a GENERADO
    }


    $tax_rate = filter_var($data['tax_rate'] ?? 0.00, FILTER_VALIDATE_FLOAT);
    $items = $data['items']; // Array de [description, unit_price, quantity]

    // Calcular subtotal y total_amount antes de insertar
    $subtotal = 0;
    foreach ($items as $item) {
        if (!isset($item[0], $item[1], $item[2])) { // description, unit_price, quantity
            throw new Exception('Formato de ítem inválido. Cada ítem debe tener descripción, precio y cantidad.');
        }
        $item_price = filter_var($item[1], FILTER_VALIDATE_FLOAT);
        $item_quantity = filter_var($item[2], FILTER_VALIDATE_INT);
        if ($item_price === false || $item_quantity === false || $item_quantity <= 0) {
            throw new Exception('Precio o cantidad de ítem inválido.');
        }
        $subtotal += ($item_price * $item_quantity);
    }
    $total_amount = $subtotal + ($subtotal * ($tax_rate / 100));

    $pdo->beginTransaction();

    // Generar un external_id único (similar a como se hace con leads)
    $external_id = 'est-' . uniqid();

    // Insertar el estimado principal
    $stmt = $pdo->prepare(
        "INSERT INTO estimates (external_id, client_id, lead_id, estimate_date, status, tax_rate, subtotal, total_amount)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)"
    );
    $stmt->execute([
        $external_id,
        $client_id,
        $lead_id,
        $estimate_date,
        $status, // Usar el estado ajustado (generado)
        $tax_rate,
        $subtotal,
        $total_amount
    ]);

    $estimate_id = $pdo->lastInsertId();

    // Insertar los ítems del estimado
    $stmt_items = $pdo->prepare(
        "INSERT INTO estimate_items (estimate_id, description, unit_price, quantity)
         VALUES (?, ?, ?, ?)"
    );
    foreach ($items as $item) {
        $stmt_items->execute([
            $estimate_id,
            $item[0], // description
            filter_var($item[1], FILTER_VALIDATE_FLOAT), // unit_price
            filter_var($item[2], FILTER_VALIDATE_INT) // quantity
        ]);
    }

    $pdo->commit();

    $response = [
        'success' => true,
        'message' => 'Estimado agregado con éxito.',
        'id' => $estimate_id,
        'external_id' => $external_id
    ];

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("PDO Error en quote-create.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => 'Error de Base de Datos: ' . $e->getMessage()];
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("General Error en quote-create.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => $e->getMessage()];
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>