<?php
// db/quote-update.php
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

include_once 'db_connection.php';

$response = [];

try {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error: JSON inválido recibido. ' . json_last_error_msg());
    }

    if (empty($data['id'])) {
        throw new Exception('ID del estimado no proporcionado para la actualización.');
    }

    $estimate_id = filter_var($data['id'], FILTER_VALIDATE_INT);
    if ($estimate_id === false) {
        throw new Exception('ID de estimado inválido.');
    }

    // --- NUEVA LÓGICA: Actualización de estado solamente ---
    // Si solo se envían 'id' y 'status', actualiza solo el estado
    if (isset($data['status']) && count($data) === 2) {
        $new_status = trim($data['status']);
        $stmt = $pdo->prepare("UPDATE estimates SET status = ? WHERE id = ?");
        
        if ($stmt->execute([$new_status, $estimate_id])) {
            $response = ['success' => true, 'message' => 'Estado del estimado actualizado con éxito.'];
        } else {
            throw new Exception('Falló la ejecución de la consulta de actualización de estado.');
        }
        
        ob_end_clean();
        header('Content-Type: application/json');
        echo json_encode($response);
        exit; // Terminar ejecución aquí para actualizaciones de solo estado
    }
    // --- FIN NUEVA LÓGICA ---

    // Lógica existente para actualización completa del estimado
    // (Esta parte se ejecuta solo si se envían más campos además de 'id' y 'status')
    if (empty($data['estimate_date']) || (!isset($data['client_id']) && !isset($data['lead_id'])) || !isset($data['items'])) {
        throw new Exception('Datos incompletos para el estimado. Fecha, contacto e ítems son requeridos para la actualización completa.');
    }
    if (!is_array($data['items'])) {
        throw new Exception('Los ítems del estimado deben ser un array.');
    }

    $client_id = !empty($data['client_id']) ? filter_var($data['client_id'], FILTER_VALIDATE_INT) : null;
    $lead_id = !empty($data['lead_id']) ? trim($data['lead_id']) : null;
    $estimate_date = $data['estimate_date'];
    $status = trim($data['status']); // El estado se toma de los datos de entrada para actualización completa
    $tax_rate = filter_var($data['tax_rate'] ?? 0.00, FILTER_VALIDATE_FLOAT);
    $items = $data['items'];

    $subtotal = 0;
    foreach ($items as $item) {
        if (!isset($item[0], $item[1], $item[2])) {
            throw new Exception('Formato de ítem inválido. Cada ítem debe tener descripción, precio y cantidad.');
        }
        $item_price = filter_var($item[1], FILTER_VALIDATE_FLOAT);
        $item_quantity = filter_var($item[2], FILTER_VALIDATE_INT);
        if ($item_price === false || $item_quantity === false || $item_quantity <= 0) {
            throw new Exception('Precio o cantidad de ítem inválido.');
        }
        $subtotal += ($item_price * $item_quantity);
    }
    $total_amount = $subtotal + ($subtotal * ($tax_rate / 100));

    $pdo->beginTransaction();

    // 1. Actualizar el estimado principal
    $stmt = $pdo->prepare(
        "UPDATE estimates SET
            client_id = ?,
            lead_id = ?,
            estimate_date = ?,
            status = ?,
            tax_rate = ?,
            subtotal = ?,
            total_amount = ?
         WHERE id = ?"
    );
    $stmt->execute([
        $client_id,
        $lead_id,
        $estimate_date,
        $status,
        $tax_rate,
        $subtotal,
        $total_amount,
        $estimate_id
    ]);

    // 2. Eliminar ítems existentes y reinsertar los nuevos
    $stmt_delete_items = $pdo->prepare("DELETE FROM estimate_items WHERE estimate_id = ?");
    $stmt_delete_items->execute([$estimate_id]);

    if (!empty($items)) {
        $stmt_insert_items = $pdo->prepare(
            "INSERT INTO estimate_items (estimate_id, description, unit_price, quantity)
             VALUES (?, ?, ?, ?)"
        );
        foreach ($items as $item) {
            $stmt_insert_items->execute([
                $estimate_id,
                $item[0], // description
                filter_var($item[1], FILTER_VALIDATE_FLOAT), // unit_price
                filter_var($item[2], FILTER_VALIDATE_INT) // quantity
            ]);
        }
    }

    $pdo->commit();

    $response = ['success' => true, 'message' => 'Estimado actualizado con éxito.'];

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("PDO Error en quote-update.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => 'Error de Base de Datos: ' . $e->getMessage()];
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("General Error en quote-update.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => $e->getMessage()];
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>