<?php
// db/website-config-update.php
ob_start();
error_reporting(0); // Desactivar errores en producción
ini_set('display_errors', 0);

include_once 'db_connection.php';
include_once 'file_helpers.php'; // Incluir el nuevo archivo de helpers

$response = ['success' => false, 'message' => ''];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error de JSON en website-config-update.php: " . json_last_error_msg());
        $response['message'] = 'Error al decodificar datos de solicitud.';
        echo json_encode($response);
        exit();
    }
    if (empty($input)) {
        $response['message'] = 'Datos de solicitud vacíos o no válidos.';
        echo json_encode($response);
        exit();
    }

    // Asumimos que actualizaremos la configuración del inquilino con ID 1.
    // En un sistema real, el tenant_id vendría de la sesión del usuario autenticado.
    $tenant_id = 1; // ID del inquilino actual (usar variable dinámica en producción)

    // Campos simples
    $company_name = trim($input['company_name'] ?? '');
    $contact_name = trim($input['contact_name'] ?? '');
    $phone_primary = trim($input['phone_primary'] ?? '');
    $phone_secondary = trim($input['phone_secondary'] ?? '');
    $email = trim($input['email'] ?? '');
    $website_url = trim($input['website_url'] ?? '');
    $new_logo_url = trim($input['logo_url'] ?? null); // La nueva URL del logo

    // Campos de configuración de agenda
    $enable_booking = (int)($input['enable_booking'] ?? false); 
    $slot_duration = filter_var($input['slot_duration'] ?? 60, FILTER_VALIDATE_INT);
    $buffer_time = filter_var($input['buffer_time'] ?? 15, FILTER_VALIDATE_INT);
    $available_spots = filter_var($input['available_spots'] ?? 1, FILTER_VALIDATE_INT);


    // Campos JSON (asegurarse de que sean arrays antes de codificar)
    $new_gallery_images = $input['gallery_images'] ?? []; // Array de URLs de la nueva galería
    $gallery_images_json = json_encode($new_gallery_images); // Codificar para guardar

    $cities_operated = json_encode($input['cities_operated'] ?? []);
    $weekly_schedule = json_encode($input['weekly_schedule'] ?? []);
    $blocked_dates = json_encode($input['blocked_dates'] ?? []);


    // Validaciones de campos obligatorios
    if (empty($company_name)) {
        $response['message'] = 'El nombre del negocio es obligatorio.';
        echo json_encode($response);
        exit();
    }
    if (empty($email)) {
        $response['message'] = 'El email es obligatorio.';
        echo json_encode($response);
        exit();
    }
    if (empty($phone_primary)) {
        $response['message'] = 'El teléfono principal es obligatorio.';
        echo json_encode($response);
        exit();
    }

    try {
        $pdo->beginTransaction();

        // 1. Obtener la configuración actual (incluyendo logo_url y gallery_images antiguas)
        $stmt_old_config = $pdo->prepare("SELECT logo_url, gallery_images FROM website_config WHERE id = 1");
        $stmt_old_config->execute();
        $old_config = $stmt_old_config->fetch(PDO::FETCH_ASSOC);

        $old_logo_url = $old_config['logo_url'] ?? null;
        $old_gallery_images = json_decode($old_config['gallery_images'] ?? '[]', true);


        // 2. Actualizar la fila de configuración con los nuevos datos
        $stmt = $pdo->prepare("
            UPDATE website_config SET
                company_name = ?,
                contact_name = ?,
                phone_primary = ?,
                phone_secondary = ?,
                email = ?,
                website_url = ?,
                logo_url = ?,
                gallery_images = ?,
                cities_operated = ?,
                weekly_schedule = ?,
                enable_booking = ?,
                slot_duration = ?,
                buffer_time = ?,
                available_spots = ?,
                blocked_dates = ?
            WHERE id = 1
        ");

        $stmt->execute([
            $company_name,
            $contact_name,
            $phone_primary,
            $phone_secondary,
            $email,
            $website_url,
            $new_logo_url, // Usar la nueva URL
            $gallery_images_json, // Usar las nuevas imágenes de galería JSON
            $cities_operated,
            $weekly_schedule,
            $enable_booking,
            $slot_duration,
            $buffer_time,
            $available_spots,
            $blocked_dates
        ]);

        // Si la fila no existía (muy raro si se usó INSERT IGNORE), podemos insertarla
        if ($stmt->rowCount() === 0) {
            $stmt_insert = $pdo->prepare("
                INSERT INTO website_config (id, company_name, contact_name, phone_primary, phone_secondary, email, website_url, logo_url, gallery_images, cities_operated, weekly_schedule, enable_booking, slot_duration, buffer_time, available_spots, blocked_dates)
                VALUES (1, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt_insert->execute([
                $company_name,
                $contact_name,
                $phone_primary,
                $phone_secondary,
                $email,
                $website_url,
                $new_logo_url,
                $gallery_images_json,
                $cities_operated,
                $weekly_schedule,
                $enable_booking,
                $slot_duration,
                $buffer_time,
                $available_spots,
                $blocked_dates
            ]);
        }

        $pdo->commit();

        // 3. Eliminar archivos antiguos DEL SERVIDOR DESPUÉS de la actualización de la DB
        // Eliminar logo antiguo si ha cambiado
        if (!empty($old_logo_url) && $old_logo_url !== $new_logo_url) {
            deleteFileFromServer($old_logo_url, $tenant_id);
        }

        // Eliminar imágenes de galería antiguas que ya no estén en la nueva lista
        foreach ($old_gallery_images as $old_img_url) {
            if (!in_array($old_img_url, $new_gallery_images)) {
                deleteFileFromServer($old_img_url, $tenant_id);
            }
        }

        $response['success'] = true;
        $response['message'] = 'Configuración de la web actualizada con éxito.';

    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error PDO al actualizar configuración web: " . $e->getMessage());
        $response['message'] = 'Error de Base de Datos al actualizar configuración: ' . $e->getMessage();
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error general al actualizar configuración web: " . $e->getMessage());
        $response['message'] = 'Error interno al actualizar configuración: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Método de solicitud no válido.';
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit();
?>